<?php
declare(strict_types=1);

/**
 * PRODUCTION-GRADE WOOCOMMERCE WEBHOOK ENGINE
 * 
 * STRICT SECURITY RULES:
 * - No output before headers.
 * - No debug echo/print/var_dump.
 * - Validate signature BEFORE processing.
 * - PDO prepared statements only.
 * - Try/catch all DB operations.
 */

// 1. Initial configuration & Security
define('WOO_WEBHOOK_SECRET', 'your-secret');
define('WEBHOOK_LOG_FILE', __DIR__ . '/../logs/webhook.log');

function log_error(string $msg)
{
    if (!is_dir(dirname(WEBHOOK_LOG_FILE))) {
        @mkdir(dirname(WEBHOOK_LOG_FILE), 0755, true);
    }
    $timestamp = date('Y-m-d H:i:s');
    @file_put_contents(WEBHOOK_LOG_FILE, "[$timestamp] ERROR: $msg\n", FILE_APPEND);
}

// 2. Read raw payload
$payload = file_get_contents('php://input');

// 3. Read header
$signature = $_SERVER['HTTP_X_WC_WEBHOOK_SIGNATURE'] ?? '';

// 4. Validate signature IMMEDIATELY
$expected = base64_encode(hash_hmac('sha256', $payload, WOO_WEBHOOK_SECRET, true));
if (!hash_equals($expected, $signature)) {
    http_response_code(401);
    exit;
}

// 5. Decode JSON safely
$data = json_decode($payload, true);
if (!$data || !isset($data['id'])) {
    http_response_code(400);
    exit;
}

// 6. Extract required fields
$order_id = (string)$data['id'];
$status = (string)$data['status'];
$email = $data['billing']['email'] ?? null;
$total = $data['total'] ?? null;
$currency = $data['currency'] ?? null;

// License Generation Logic
function generateLicense()
{
    return 'BIO-' .
        strtoupper(substr(bin2hex(random_bytes(4)), 0, 4)) .
        '-' .
        strtoupper(substr(bin2hex(random_bytes(4)), 0, 4));
}

// 7. Database Operations
require_once __DIR__ . '/../includes/db.php';

try {
    // A. Insert raw payload into order_logs table
    $stmt = $pdo->prepare("INSERT INTO order_logs (woo_order_id, status, raw_payload) VALUES (?, ?, ?)");
    $stmt->execute([$order_id, $status, $payload]);

    // B. Insert or update orders table
    $stmt = $pdo->prepare("INSERT INTO orders (woo_order_id, customer_email, amount, currency, status, updated_at) 
                           VALUES (?, ?, ?, ?, ?, CURRENT_TIMESTAMP)
                           ON CONFLICT(woo_order_id) DO UPDATE SET 
                           status = excluded.status, updated_at = CURRENT_TIMESTAMP");
    $stmt->execute([$order_id, $email, $total, $currency, $status]);

    // C. Logic for 'completed' orders
    if ($status === 'completed') {

        // Start transaction for atomic license creation
        $pdo->beginTransaction();

        // Check for duplicate in processed_orders
        $check = $pdo->prepare("SELECT 1 FROM processed_orders WHERE woo_order_id = ?");
        $check->execute([$order_id]);

        if (!$check->fetch()) {
            // Generate license string
            $license = generateLicense();

            // Insert into licenses table
            $stmt = $pdo->prepare("INSERT INTO licenses (license_key, client_email, status, created_at) VALUES (?, ?, 'active', CURRENT_TIMESTAMP)");
            $stmt->execute([$license, $email]);

            // Update orders.license_key
            $stmt = $pdo->prepare("UPDATE orders SET license_key = ? WHERE woo_order_id = ?");
            $stmt->execute([$license, $order_id]);

            // Track processed order
            $stmt = $pdo->prepare("INSERT INTO processed_orders (woo_order_id) VALUES (?)");
            $stmt->execute([$order_id]);

            $pdo->commit();

            // Trigger PHPMailer - Only after commit success
            if (file_exists(__DIR__ . '/../../libs/PHPMailer/src/PHPMailer.php')) {
                require_once __DIR__ . '/../../libs/PHPMailer/src/Exception.php';
                require_once __DIR__ . '/../../libs/PHPMailer/src/PHPMailer.php';
                require_once __DIR__ . '/../../libs/PHPMailer/src/SMTP.php';

                $stmtSettings = $pdo->query("SELECT * FROM settings WHERE id = 1 LIMIT 1");
                $settings = $stmtSettings->fetch();

                if ($settings) {
                    try {
                        $mail = new \PHPMailer\PHPMailer\PHPMailer(true);
                        $mail->isSMTP();
                        $mail->Host = $settings['smtp_host'];
                        $mail->SMTPAuth = true;
                        $mail->Username = $settings['smtp_user'];
                        $mail->Password = $settings['smtp_pass'];
                        $mail->SMTPSecure = (int)$settings['smtp_port'] === 465 ?\PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_SMTPS : \PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS;
                        $mail->Port = (int)$settings['smtp_port'];
                        $mail->setFrom($settings['smtp_from_email'], $settings['smtp_from_name']);
                        $mail->addAddress($email);
                        $mail->isHTML(true);
                        $mail->Subject = 'Your BioScript Architecture License Key';
                        $mail->Body = "Your new license key is: <strong>{$license}</strong>";
                        $mail->send();
                    }
                    catch (Exception $e) {
                        log_error("Email failed for order $order_id: " . $e->getMessage());
                    }
                }
            }
        }
        else {
            // Already processed
            $pdo->rollBack();
        }
    }

    // D. Logic for 'refunded' orders
    if ($status === 'refunded') {

        $pdo->beginTransaction();

        // Find linked license from orders
        $stmt = $pdo->prepare("SELECT license_key FROM orders WHERE woo_order_id = ?");
        $stmt->execute([$order_id]);
        $linked_license = $stmt->fetchColumn();

        if ($linked_license) {
            // Update licenses.status = 'revoked'
            $stmt = $pdo->prepare("UPDATE licenses SET status = 'revoked' WHERE license_key = ?");
            $stmt->execute([$linked_license]);

            // Update orders.status = 'refunded' (redundant with the upsert above but fulfills explicit flow rule 9)
            $stmt = $pdo->prepare("UPDATE orders SET status = 'refunded' WHERE woo_order_id = ?");
            $stmt->execute([$order_id]);
        }

        $pdo->commit();
    }

}
catch (PDOException $e) {
    if ($pdo && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    log_error("DB Error on order $order_id: " . $e->getMessage());
}

// 8. Final exit
http_response_code(200);
echo "OK";
exit;